<?php

namespace App\Models;

use CodeIgniter\Model;

class AsistenciaModel extends Model
{
    protected $table = 'asistencias';
    protected $primaryKey = 'id_asistencia';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'id_estudiante',
        'id_materia',          // ✅ AGREGADO
        'fecha_asistencia',
        'hora_registro',
        'estado_asistencia',
        'observaciones',
        'registrado_por'
    ];

    // Dates
    protected $useTimestamps = false;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Validation
    protected $validationRules = [
        'id_estudiante' => 'required|integer',
        'id_materia' => 'required|integer',  // ✅ AGREGADO
        'fecha_asistencia' => 'required|valid_date',
        'estado_asistencia' => 'required|in_list[presente,ausente,tardanza]'
    ];

    protected $validationMessages = [
        'id_estudiante' => [
            'required' => 'El ID del estudiante es requerido',
            'integer' => 'El ID del estudiante debe ser un número'
        ],
        'id_materia' => [
            'required' => 'El ID de la materia es requerido',
            'integer' => 'El ID de la materia debe ser un número'
        ],
        'fecha_asistencia' => [
            'required' => 'La fecha es requerida',
            'valid_date' => 'La fecha debe ser válida'
        ],
        'estado_asistencia' => [
            'required' => 'El estado es requerido',
            'in_list' => 'El estado debe ser: presente, ausente o tardanza'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Obtener asistencias con detalles de estudiantes y materias
     * 
     * @param string|null $fechaInicio
     * @param string|null $fechaFin
     * @param int|null $idMateria
     * @return array
     */
    public function obtenerAsistenciasConDetalles($fechaInicio = null, $fechaFin = null, $idMateria = null)
    {
        $builder = $this->db->table($this->table);
        
        $builder->select('
            asistencias.*,
            estudiantes.codigo_estudiante,
            estudiantes.nombres,
            estudiantes.apellidos,
            estudiantes.email,
            materias.nombre_materia,
            materias.grado,
            materias.seccion
        ');
        
        $builder->join('estudiantes', 'estudiantes.id_estudiante = asistencias.id_estudiante', 'left');
        $builder->join('materias', 'materias.id_materia = asistencias.id_materia', 'left');
        
        // Filtros opcionales
        if ($fechaInicio) {
            $builder->where('asistencias.fecha_asistencia >=', $fechaInicio);
        }
        
        if ($fechaFin) {
            $builder->where('asistencias.fecha_asistencia <=', $fechaFin);
        }
        
        // ✅ NUEVO - Filtro por materia
        if ($idMateria) {
            $builder->where('asistencias.id_materia', $idMateria);
        }
        
        $builder->orderBy('asistencias.fecha_asistencia', 'DESC');
        $builder->orderBy('asistencias.hora_registro', 'DESC');
        
        return $builder->get()->getResultArray();
    }

    /**
     * Obtener asistencias de un estudiante específico
     * 
     * @param int $idEstudiante
     * @param int|null $idMateria
     * @return array
     */
    public function obtenerAsistenciasEstudiante($idEstudiante, $idMateria = null)
    {
        $builder = $this->db->table($this->table);
        
        $builder->select('
            asistencias.*,
            materias.nombre_materia,
            materias.grado,
            materias.seccion
        ');
        
        $builder->join('materias', 'materias.id_materia = asistencias.id_materia', 'left');
        $builder->where('asistencias.id_estudiante', $idEstudiante);
        
        // ✅ NUEVO - Filtro por materia
        if ($idMateria) {
            $builder->where('asistencias.id_materia', $idMateria);
        }
        
        $builder->orderBy('asistencias.fecha_asistencia', 'DESC');
        
        return $builder->get()->getResultArray();
    }

    /**
     * Obtener asistencias por fecha
     * 
     * @param string $fecha
     * @param int|null $idMateria
     * @return array
     */
    public function obtenerAsistenciasPorFecha($fecha, $idMateria = null)
    {
        $builder = $this->db->table($this->table);
        
        $builder->select('
            asistencias.*,
            estudiantes.codigo_estudiante,
            estudiantes.nombres,
            estudiantes.apellidos,
            materias.nombre_materia
        ');
        
        $builder->join('estudiantes', 'estudiantes.id_estudiante = asistencias.id_estudiante', 'left');
        $builder->join('materias', 'materias.id_materia = asistencias.id_materia', 'left');
        $builder->where('asistencias.fecha_asistencia', $fecha);
        
        // ✅ NUEVO - Filtro por materia
        if ($idMateria) {
            $builder->where('asistencias.id_materia', $idMateria);
        }
        
        $builder->orderBy('estudiantes.apellidos', 'ASC');
        
        return $builder->get()->getResultArray();
    }

    /**
     * ✅ NUEVO - Obtener asistencias por materia
     * 
     * @param int $idMateria
     * @param string|null $fecha
     * @return array
     */
    public function obtenerAsistenciasPorMateria($idMateria, $fecha = null)
    {
        $builder = $this->db->table($this->table);
        
        $builder->select('
            asistencias.*,
            estudiantes.codigo_estudiante,
            estudiantes.nombres,
            estudiantes.apellidos,
            estudiantes.email
        ');
        
        $builder->join('estudiantes', 'estudiantes.id_estudiante = asistencias.id_estudiante', 'left');
        $builder->where('asistencias.id_materia', $idMateria);
        
        if ($fecha) {
            $builder->where('asistencias.fecha_asistencia', $fecha);
        }
        
        $builder->orderBy('asistencias.fecha_asistencia', 'DESC');
        $builder->orderBy('estudiantes.apellidos', 'ASC');
        
        return $builder->get()->getResultArray();
    }

    /**
     * Verificar si existe asistencia
     * 
     * @param int $idEstudiante
     * @param int $idMateria
     * @param string $fecha
     * @return bool
     */
    public function existeAsistencia($idEstudiante, $idMateria, $fecha)
    {
        return $this->where('id_estudiante', $idEstudiante)
                    ->where('id_materia', $idMateria)
                    ->where('fecha_asistencia', $fecha)
                    ->first() !== null;
    }

    public function getAsistenciasPorFecha($fecha, $idMateria = null)
    {
        $builder = $this->db->table('asistencias a');
        $builder->select('a.*, 
                        e.codigo_estudiante, 
                        e.nombres, 
                        e.apellidos, 
                        e.email,
                        m.nombre_materia,
                        m.grado,
                        m.seccion');
        $builder->join('estudiantes e', 'a.id_estudiante = e.id_estudiante');
        $builder->join('materias m', 'a.id_materia = m.id_materia', 'left');
        $builder->where('a.fecha_asistencia', $fecha);
        
        if ($idMateria !== null) {
            $builder->where('a.id_materia', $idMateria);
        }
        
        $builder->orderBy('a.hora_registro', 'DESC');
        
        return $builder->get()->getResultArray();
    }
}

